package lejos.realtime;

import lejos.nxt.comm.RConsole;

/**
 * This {@code ReleaseParameters} class is an equivalent of the
 * {@code ReleaseParameters} class in RTSJ.
 * 
 * @author Team iRboT
 */
public class ReleaseParameters extends SchedulingParameters {
//    private HighResolutionTime start;
//    private RelativeTime period;

    private RelativeTime cost;
    private RelativeTime deadline;
    private AsyncEventHandler costOverrunHandler;
    private AsyncEventHandler deadlineMissHandler;

    /**
     * Create a new instance of ReleaseParameters.
     * 
     * This constructor creates a default ReleaseParameters object, i.e., it is
     * equivalent to ReleaseParameters(null, null, null, null).
     */
    public ReleaseParameters() {
        this(null, null, null, null);

    }

    /**
     * Create a new instance of ReleaseParameters with the given parameter
     * values.
     *
     * @param cost
     *          Processing time units per release. On implementations which can
     * measure the amount of time a schedulable object is executed, this value
     * is the maximum amount of time a schedulable object receives per release.
     * On implementations which cannot measure execution time, this value is
     * used as a hint to the feasibility algorithm. On such systems it is not
     * possible to determine when any particular object exceeds cost. If null,
     * the default value is a new instance of RelativeTime(0,0).
     *
     * @param deadline
     *          The latest permissible completion time measured from the
     * release time of the associated invocation of the schedulable object.
     * There is no default for deadline in this class. The default must be
     * determined by the subclasses.
     *
     * @param overrunHandler
     *          This handler is invoked if an invocation of the schedulable
     * object exceeds cost. In the minimum implementation overrunHandler is
     * ignored. If null, no application event handler is executed on cost
     * overrun.
     *
     * @param missHandler
     *          This handler is invoked if the run() method of the schedulable
     * object is still executing after the deadline has passed. Although
     * minimum implementations do not consider deadlines in feasibility
     * calculations, they must recognize variable deadlines and invoke the
     * miss handler as appropriate. If null, no application event handler is
     * executed on the miss deadline condition.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the time value of cost is less than zero, or the
     * time value of deadline is less than or equal to zero.
     *
     * @throws IllegalAssignmentError
     *          Thrown if cost, deadline, overrunHandler, or missHandler cannot
     * be stored in this.
     *
     */
    public ReleaseParameters(RelativeTime cost, RelativeTime deadline, AsyncEventHandler overrunHandler, AsyncEventHandler missHandler) {




        setCost(cost);
        setDeadline(deadline);
        setCostOverrunHandler(overrunHandler);
        setDeadlineMissHandler(missHandler);
        
    }

    /**
     * Return a clone of this.
     *
     * This method should behave effectively as if it constructed a new object
     * with clones of the high-resolution time values of this.
     * The new object is in the current allocation context clone does not copy
     * any associations from this and it does not implicitly bind the new
     * object to a SO.
     * The new object has clones of all high-resolution time values (deep copy).
     * References to event handlers are copied (shallow copy.)
     */
    @Override
    public Object clone() {
        return new ReleaseParameters(cost, deadline, costOverrunHandler, deadlineMissHandler);
    }

    public RelativeTime getCost() {
        return cost;
    }

    public void setCost(RelativeTime cost) {
        
        if (cost == null) {
            //new RelativeTime(0,0) matching with infinite time for deadline or cost
            cost = new RelativeTime(0, 0);
        }
        if (cost.getMilliseconds() < 0) {

            throw new IllegalArgumentException("Cost must be greater than 0.");

        }
        this.cost = cost;
    }

    public RelativeTime getDeadline() {
        return deadline;
    }

    public void setDeadline(RelativeTime deadline) {
        if(deadline == null){
            //new RelativeTime(0,0) matching with infinite time for deadline or cost
            deadline = new RelativeTime(0,0);

        }
        if ( deadline.getMilliseconds() < 0) {

            throw new IllegalArgumentException("Deadline must be greater than zero.");
        }
        this.deadline = deadline;

    }

    public AsyncEventHandler getDeadlineMissHandler() {
        return deadlineMissHandler;
    }

    public void setDeadlineMissHandler(AsyncEventHandler missHandler) {
        this.deadlineMissHandler = missHandler;
    }

    public AsyncEventHandler getCostOverrunHandler() {
        return costOverrunHandler;
    }

    public void setCostOverrunHandler(AsyncEventHandler overrunHandler) {
        this.costOverrunHandler = overrunHandler;
    }

    /**
     * This method first performs a feasibility analysis using the new cost,
     * and deadline as replacements for the matching attributes of all
     * schedulable objects associated with this release parameters object.
     *
     * If the resulting system is feasible, the method replaces the current
     * scheduling characteristics of this release parameters object with the
     * new scheduling characteristics. The change in the release
     * characteristics, including the timing of the change, of any associated
     * schedulable objects will take place under the control of their
     * schedulers.
     *
     * @param cost
     *          The proposed cost. Equivalent to RelativeTime(0,0) if null.
     * (A new instance of RelativeTime is created in the memory area containing
     * this ReleaseParameters instance). If null, the default value is a new
     * instance of RelativeTime(0,0).
     *
     * @param deadline
     *          The proposed deadline. If null, the default value is new
     * instance of RelativeTime(period).
     *
     * @return
     *          True, if the resulting system is feasible and the changes are
     * made. False, if the resulting system is not feasible and no changes are
     * made.
     *
     * @throws IllegalArgumentException
     *          Thrown if the time value of cost is less than zero, or the time
     * value of deadline is less than or equal to zero.
     */
    public boolean setIfFeasible(RelativeTime cost, RelativeTime deadline) {

        throw new UnsupportedOperationException("Not overrided yet.");
    }
}
