package lejos.realtime;

import java.util.ArrayList;

/**
 *
 * @author Team iRboT
 */
public class PriorityScheduler extends Scheduler {

    protected final ArrayList<Schedulable> feasabilityList;
    protected static PriorityScheduler current = null;
    protected final String policy;

    protected PriorityScheduler(){
       feasabilityList = new ArrayList<Schedulable>();
       policy = "PriorityScheduler";
    }
    
    protected PriorityScheduler(String policy){
       feasabilityList = new ArrayList<Schedulable>();
       this.policy = policy;
    }

    /**
     * Return a reference to the distinguished instance of PriorityScheduler
     * which is the system's base scheduler.
     *
     * @return A reference to the distinguished instance PriorityScheduler.
     */
    public static PriorityScheduler instance(){
        if(current == null){
            current = new PriorityScheduler();
        }
        return current;
    }

    /**
     * Gets the maximum priority available for a schedulable object managed by
     * this scheduler.
     *
     * @return The value of the maximum priority.
     */
    public int getMaxPriority(){
        return RealtimeThread.MAX_PRIORITY;
    }

    /**
     * Gets the maximum priority for the given thread. If the given thread is a
     * real-time thread that is scheduled by an instance of PriorityScheduler,
     * then the maximum priority for that scheduler is returned. If the given
     * thread is a Java thread then the maximum priority of its thread group is
     * returned. Otherwise an exception is thrown.
     *
     * @param thread - An instance of Thread. If null, the maximum priority of
     * this scheduler is returned.
     *
     * @return The maximum priority for thread
     *
     * @throws java.lang.IllegalArgumentException - Thrown if thread is a
     * real-time thread that is not scheduled by an instance of
     * PriorityScheduler.
     */
    public static int getMaxPriority(java.lang.Thread thread){
        return thread.MAX_PRIORITY;
    }

    /**
     * Gets the minimum priority available for a schedulable object managed
     * by this scheduler.
     *
     * @return The minimum priority used by this scheduler.
     */
    public int getMinPriority(){
        return RealtimeThread.MIN_PRIORITY;
    }

    /**
     * Gets the minimum priority for the given thread. If the given thread is a
     * real-time thread that is scheduled by an instance of PriorityScheduler,
     * then the minimum priority for that scheduler is returned. If the given
     * thread is a Java thread then Thread.MIN_PRIORITY is returned. Otherwise
     * an exception is thrown.
     *
     * @param thread - An instance of Thread. If null, the minimum priority of
     * this scheduler is returned.
     * @return The minimum priority for thread
     * @throws java.lang.IllegalArgumentException - Thrown if thread is a
     * real-time thread that is not scheduled by an instance of
     * PriorityScheduler.
     */
    public static int getMinPriority(java.lang.Thread thread){
        return thread.MIN_PRIORITY;
    }

    /**
     * Gets the normal priority available for a schedulable object managed by
     * this scheduler.
     *
     * @return The value of the normal priority.
     */
    public int getNormPriority(){
        return RealtimeThread.NORM_PRIORITY;
    }

    /**
     * Gets the "norm" priority for the given thread. If the given thread is a
     * real-time thread that is scheduled by an instance of PriorityScheduler,
     * then the norm priority for that scheduler is returned. If the given
     * thread is a Java thread then Thread.NORM_PRIORITY is returned. Otherwise
     * an exception is thrown.
     *
     * @param thread - An instance of Thread. If null, the norm priority for
     * this scheduler is returned.
     * @return The norm priority for thread
     * @throws java.lang.IllegalArgumentException - Thrown if thread is a
     * real-time thread that is not scheduled by an instance3 of
     * PriorityScheduler.
     */
    public static int getNormPriority(java.lang.Thread thread){
        return thread.NORM_PRIORITY;
    }

    /**
     * Queries this scheduler about the feasibility of the set of schedulable
     * objects currently in the feasibility set.
     *
     * Implementation Notes:
     * The default feasibility test for the PriorityScheduler considers a set of
     * schedulable objects with bounded resource requirements, to always be
     * feasible. This covers all schedulable objects with release parameters of
     * types PeriodicParameters and SporadicParameters.
     *
     * If any schedulable object within the feasibility set has release
     * parameters of the exact type AperiodicParameters (not a subclass
     * thereof), then the feasibility set is not feasible, as aperiodic release
     * characteristics require unbounded resources. In that case, this method
     * will return false and all methods in the setIfFeasible family of methods
     * will also return false. Consequently, any call to a setIfFeasible method
     * that passes a schedulable object which has release parameters of type
     * AperiodicParameters, or passes proposed release parameters of type
     * AperiodicParameters, will return false. The only time a setIfFeasible
     * method can return true, when there exists in the feasibility set a
     * schedulable object with release parameters of type AperiodicParameters,
     * is when the method will change those release parameters to not be
     * AperiodicParameters.
     *
     * Implementations may provide a feasibility test other than the default
     * test just described. In which case the details of that test should be
     * documented here in place of this description of the default
     * implementation.
     * Specified by:
     * isFeasible in class @see Scheduler
     * @return True, if the system is feasible. False, if not.
     */
    public boolean isFeasible(){
        ArrayList[] allThreads = new ArrayList[getMaxPriority()+1];
        for(int i =0;i<allThreads.length;i++){
            allThreads[i]= new ArrayList<RealtimeThread>();
        }
        float u = (float) 0;
        int n = feasabilityList.size();
        if(n==0){
            return true;
        }
        int wt = 0;
        int t = 1;

        for(Schedulable schedulable : feasabilityList){
            if(schedulable instanceof RealtimeThread){
                RealtimeThread rtThread = (RealtimeThread) schedulable;
                int priority = getThreadPriority(rtThread);
                allThreads[priority].add(rtThread);

                /*
                 * default DM
                 */

                float utmp = (float) getThreadCost(rtThread)/getThreadDeadline(rtThread);
                u = u + utmp;
            }
        }
        if(u>1){
            return false;
        }
        else{
            if(u<n*(Math.pow(2, (double)1/n)-1)){
                return true;
            }
            else{
                for(Schedulable s : feasabilityList){
                    int priority = 0;
                    if(s instanceof RealtimeThread){
                        /*Busy Period*/
                        RealtimeThread rtThread = (RealtimeThread) s;
                        priority = getThreadPriority(rtThread);
                        while(wt != t){
                            t = wt;
                            wt = getWorkTime(allThreads, priority, t);
                        }
                         /*end Busy Period*/

                        /*instances to study*/
                        int q = (int) Math.ceil(wt/getThreadPeriod(rtThread));

                        /*response time analysis*/
                        t=1;
                        wt=0;
                        int[] ct = new int[q];
                        for(int m=1;m<=q;m++){
                            while(wt != t){
                                t = wt;
                                wt = getWorkTimeForResponseTime(allThreads, priority, t, rtThread, m);
                            }
                            ct[m-1] = wt - m * (int) (getThreadPeriod(rtThread));
                        }
                        int r = ct[0];
                        for(int i = 1;i<q;i++){
                            if(ct[i]>r){
                                r = ct[i];
                            }
                        }
                        if(r>getThreadDeadline(rtThread)){
                            return false;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * This method first performs a feasibility analysis using the proposed
     * parameter objects as replacements for the current parameters of
     * schedulable. If the resulting system is feasible, this method replaces
     * the current parameters of schedulable with the proposed ones.
     *
     * This method does not require that the schedulable object be in the
     * feasibility set before it is called. If it is not initially a member of
     * the feasibility set it will be added if the resulting system is feasible.
     *
     * Specified by:
     * setIfFeasible in class @see Scheduler
     *
     * @param schedulable - The schedulable object for which the changes are
     * proposed.
     *
     * @param release - The proposed release parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @param memory - The proposed memory parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @return True, if the resulting system is feasible and the changes are
     * made. False, if the resulting system is not feasible and no changes are
     * made.
     *
     * @throws java.lang.IllegalArgumentException - Thrown if schedulable is
     * null, or schedulable is not associated with this scheduler, or the
     * proposed parameters are not compatible with this scheduler.
     *
     * @throws IllegalAssignmentError - Thrown if schedulable cannot hold
     * references to  the proposed parameter objects, or the parameter objects
     * cannot hold a reference to schedulable.
     *
     * @throws java.lang.IllegalThreadStateException - Thrown if the new release
     * parameters change schedulable from periodic scheduling to some other
     * protocol and schedulable is currently waiting for the next release in
     * RealtimeThread.waitForNextPeriod() or
     * RealtimeThread.waitForNextPeriodInterruptible().
     *
     */
    public boolean setIfFeasible(Schedulable schedulable, ReleaseParameters release, MemoryParameters memory){
        boolean ret = addToFeasibility(schedulable);
        if(!ret){
            removeFromFeasibility(schedulable);
        }
        return ret;
    }

    /**
     * This method first performs a feasibility analysis using the proposed
     * parameter objects as replacements for the current parameters of
     * schedulable. If the resulting system is feasible, this method replaces
     * the current parameters of schedulable with the proposed ones.
     *
     * This method does not require that the schedulable object be in the
     * feasibility set before it is called. If it is not initially a member of
     * the feasibility set it will be added if the resulting system is feasible.
     *
     * Specified by: setIfFeasible in class @see Scheduler
     *
     * @param schedulable - The schedulable object for which the changes are
     * proposed.
     *
     * @param release - The proposed release parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @param memory - The proposed memory parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @param group  - The proposed processing group parameters. If null, the
     * default value of this scheduler is used (a new object is created if the
     * default value is not null). (See PriorityScheduler.)
     *
     * @return True, if the resulting system is feasible and the changes are
     * made. False, if the resulting system is not feasible and no changes are
     * made.
     *
     * @throws java.lang.IllegalArgumentException - Thrown if schedulable is
     * null, or schedulable is not associated with this scheduler, or the
     * proposed parameters are not compatible with this scheduler.
     * @throws IllegalAssignmentError - Thrown if schedulable cannot hold
     * references to the proposed parameter objects, or the parameter objects
     * cannot hold a reference to schedulable.
     * @throws java.lang.IllegalThreadStateException - Thrown if the new release
     * parameters change schedulable from periodic scheduling to some other
     * protocol and schedulable is currently waiting for the next release in
     * RealtimeThread.waitForNextPeriod() or
     * RealtimeThread.waitForNextPeriodInterruptible().
     */
    public boolean setIfFeasible(Schedulable schedulable, ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group){
        boolean ret = addToFeasibility(schedulable);
        if(!ret){
            removeFromFeasibility(schedulable);
        }
        return ret;
    }

    /**
     *
     * This method first performs a feasibility analysis using the proposed
     * parameter objects as replacements for the current parameters of
     * schedulable. If the resulting system is feasible, this method replaces
     * the current parameters of schedulable with the proposed ones.
     *
     * This method does not require that the schedulable object be in the
     * feasibility set before it is called. If it is not initially a member of
     * the feasibility set it will be added if the resulting system is feasible.
     *
     * Specified by: setIfFeasible in class @see Scheduler
     *
     * @param schedulable - The schedulable object for which the changes are
     * proposed.
     *
     * @param scheduling - The proposed scheduling parameters. If null, the
     * default value of this scheduler is used (a new object is created if the
     * default value is not null). (See PriorityScheduler.)
     *
     * @param release - The proposed release parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @param memory - The proposed memory parameters. If null, the default
     * value of this scheduler is used (a new object is created if the default
     * value is not null). (See PriorityScheduler.)
     *
     * @param group - The proposed processing group parameters. If null, the
     * default value of this scheduler is used (a new object is created if the
     * default value is not null). (See PriorityScheduler.)
     *
     * @return True, if the resulting system is feasible and the changes are
     * made. False, if the resulting system is not feasible and no changes are
     * made.
     *
     * @throws java.lang.IllegalArgumentException - Thrown if schedulable is
     * null, or schedulable is not associated with this scheduler, or the
     * proposed parameters are not compatible with this scheduler.
     * @throws IllegalAssignmentError - Thrown if schedulable cannot hold
     * references to the proposed parameter objects, or the parameter objects
     * cannot hold a reference to schedulable.
     * @throws java.lang.IllegalThreadStateException - Thrown if the new release
     * parameters change schedulable from periodic scheduling to some other
     * protocol and schedulable is currently waiting for the next release in
     * RealtimeThread.waitForNextPeriod() or
     * RealtimeThread.waitForNextPeriodInterruptible().
     */
    public boolean setIfFeasible(Schedulable schedulable, SchedulingParameters scheduling, ReleaseParameters release, MemoryParameters memory, ProcessingGroupParameters group){
        boolean ret = addToFeasibility(schedulable);
        if(!ret){
            removeFromFeasibility(schedulable);
        }
        return ret;
    }

    /**
     * Inform this scheduler and cooperating facilities that the resource
     * demands of the given instance of Schedulable will be considered in the
     * feasibility analysis of the associated Scheduler until further notice.
     * Whether the resulting system is feasible or not, the addition is
     * completed. If the object is already included in the feasibility set, do
     * nothing.
     * Specified by: addToFeasibility in class
     * @see Scheduler
     * @param schedulable - A reference to the given instance of Schedulable
     * @return True, if the system is feasible after the addition.
     * False, if not.
     * @throws java.lang.IllegalArgumentException - Thrown if schedulable is
     * null, or if schedulable is not associated with this; that is
     * schedulable.getScheduler() != this.
     */
    protected boolean addToFeasibility(Schedulable schedulable){
        feasabilityList.add(schedulable);
        return isFeasible();
    }

    /**
     * Inform this scheduler and cooperating facilities that the resource
     * demands of the given instance of Schedulable should no longer be
     * considered in the feasibility analysis of the associated Scheduler.
     * Whether the resulting system is feasible or not, the removal is
     * completed.
     * Specified by:
     * removeFromFeasibility in class
     * @see Scheduler
     * @param schedulable - A reference to the given instance of Schedulable
     * @return True, if the removal was successful.
     * False, if the schedulable object cannot be removed from the scheduler's
     * feasibility set; e.g., the schedulable object is not part of the
     * scheduler's feasibility set.
     * @throws java.lang.IllegalArgumentException - Thrown if schedulable is null.
     */
    protected boolean removeFromFeasibility(Schedulable schedulable){
        if(schedulable == null){
            throw new IllegalArgumentException("This schedulable object is null");
        }
        return feasabilityList.remove(schedulable);
    }

    /**
     *  Trigger the execution of a schedulable object (like an
     * AsyncEventHandler).
     *
     * Specified by: fireSchedulable in class @see Scheduler
     *
     * @param schedulable  - The schedulable object to make active. If null,
     * nothing happens.
     *
     * @throws java.lang.UnsupportedOperationException - Thrown in all cases by
     * the PriorityScheduler
     */
    public void fireSchedulable(Schedulable schedulable){
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     *  Gets the policy name of this.
     *
     * Specified by: getPolicyName in class @see Scheduler
     *
     * @return The policy name (Fixed Priority) as a string.
     */
    public java.lang.String getPolicyName(){
        return policy;
    }

    protected static int getThreadPriority(RealtimeThread rt){
        if(rt.getSchedulingParameters() instanceof PriorityParameters){
            PriorityParameters priorityParams = (PriorityParameters)rt.getSchedulingParameters();
            return priorityParams.getPriority();
        }
        throw new IllegalArgumentException("No priority available for this thread");
    }

    protected static long getThreadCost(RealtimeThread rt){
        long cost = rt.getReleaseParameters().getCost().getNanoseconds()+rt.getReleaseParameters().getCost().getMilliseconds();
        return cost;
    }

    protected static long getThreadPeriod(RealtimeThread rt){
        if(rt.getReleaseParameters() instanceof PeriodicParameters){
            PeriodicParameters periodParams = (PeriodicParameters) rt.getReleaseParameters();
            long period = periodParams.getPeriod().getNanoseconds()+periodParams.getPeriod().getMilliseconds();
            return period;
        }
        throw new IllegalArgumentException("No PeriodicParameters for this RealtimeThread");
    }

    protected static long getThreadDeadline(RealtimeThread rt){
        long deadline = rt.getReleaseParameters().getDeadline().getNanoseconds()+rt.getReleaseParameters().getDeadline().getMilliseconds();
        return deadline;
    }

    protected static int getWorkTime(ArrayList[] lists, int priority, int t){
        int wt = 0;
        int n = lists.length;
        while(n>=priority){
            ArrayList list=lists[n];
            for(Object o:list){
                RealtimeThread rt = (RealtimeThread)o;
                wt += Math.ceil(t/getThreadPeriod(rt))*getThreadCost(rt);
            }
            n--;
        }
        return wt;
    }

    protected static int getWorkTimeForResponseTime(ArrayList[] lists, int priority, int t, RealtimeThread rt, int m){
        int wt = 0;
        int n = lists.length;
        while(n>=priority+1){
            ArrayList list=lists[n];
            for(Object o:list){
                RealtimeThread rtCurrent = (RealtimeThread)o;
                wt += Math.ceil(t/getThreadPeriod(rtCurrent))*getThreadCost(rtCurrent);
            }
            n--;
        }
        wt += m*getThreadCost(rt);
        return wt;
    }
}
