package lejos.realtime;

/**
 *
 * @author Team iRboT
 */
public class PeriodicParameters extends ReleaseParameters {

    private HighResolutionTime start;
    private RelativeTime period;

    /**
     * Create a PeriodicParameters object.
     * This constructor has the same effect as invoking
     * {@link PeriodicParameters(null, period, null, null, null, null)}
     *
     * @param period
     *          The period is the interval between successive unblocks of the
     * {@code RealtimeThread.waitForNextPeriod()} and
     * {@code RealtimeThread.waitForNextPeriodInterruptible()} methods.
     * There is no default value. If period is null an exception is thrown.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the period is null or its time value is not greater
     * than zero.
     * 
     * @throws IllegalAssignmentError
     *          Thrown if period cannot be stored in this.
     */
    public PeriodicParameters(RelativeTime period) {
        
        this(null, period, null, null, null, null);

    }

    /**
     * Create a PeriodicParameters object.
     * This constructor has the same effect as invoking
     * {@link PeriodicParameters(start, period, null, null, null, null)}
     *
     * @param start
     *          Time at which the first release begins (i.e. the real-time
     * thread becomes eligible for execution.) If a RelativeTime, this time
     * is relative to the first time the thread becomes activated (that is,
     * when start() is called). If an AbsoluteTime, then the first release is
     * the maximum of the start parameter and the time of the call to the
     * associated RealtimeThread.start() method. If null, the default value is
     * a new instance of RelativeTime(0,0).
     *
     * @param period
     *          The period is the interval between successive unblocks of the
     * RealtimeThread.waitForNextPeriod() and
     * RealtimeThread.waitForNextPeriodInterruptible() methods. There is no
     * default value. If period is null an exception is thrown.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the period is null or its time value is not greater
     * than zero.
     * 
     * @throws IllegalAssignmentError
     *          Thrown if start or period cannot be stored in this.
     */
    public PeriodicParameters(HighResolutionTime start, RelativeTime period) {
        this(start, period, null, null, null, null);
    }

    /**
     * Create a PeriodicParameters object.
     *
     * @param start
     *          Time at which the first release begins (i.e. the real-time
     * thread becomes eligible for execution.) If a RelativeTime, this time is
     * relative to the first time the thread becomes activated (that is, when
     * start() is called). If an AbsoluteTime, then the first release is the
     * maximum of the start parameter and the time of the call to the
     * associated RealtimeThread.start() method. If null, the default value is
     * a new instance of RelativeTime(0,0).
     *
     * @param period
     *          The period is the interval between successive unblocks of the
     * RealtimeThread.waitForNextPeriod() and
     * RealtimeThread.waitForNextPeriodInterruptible() methods. There is no
     * default value. If period is null an exception is thrown.
     * 
     * @param cost
     *          Processing time per release. On implementations which can
     * measure the amount of time a schedulable object is executed, this value
     * is the maximum amount of time a schedulable object receives per release.
     * On implementations which cannot measure execution time, this value is
     * used as a hint to the feasibility algorithm. On such systems it is not
     * possible to determine when any particular object exceeds or will exceed
     * cost time units in a release. If null, the default value is a new
     * instance of RelativeTime(0,0).
     * 
     * @param deadline
     *          The latest permissible completion time measured from the
     * release time of the associated invocation of the schedulable object.
     * If null, the default value is new instance of RelativeTime(period).
     * 
     * @param overrunHandler
     *          This handler is invoked if an invocation of the schedulable
     * object exceeds cost in the given release. Implementations may ignore this
     * parameter. If null, the default value no overrun handler.
     * 
     * @param missHandler
     *          This handler is invoked if the run() method of the schedulable
     * object is still executing after the deadline has passed. Although minimum
     * implementations do not consider deadlines in feasibility calculations,
     * they must recognize variable deadlines and invoke the miss handler as
     * appropriate. If null, the default value no deadline miss handler.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the period is null or its time value is not greater
     * than zero, or if the time value of cost is less than zero, or if the
     * time value of deadline is not greater than zero.
     *
     * @throws IllegalAssignmentError
     *          Thrown if start, period, cost, deadline, overrunHandler or
     * missHandler cannot be stored in this.
     */
    public PeriodicParameters(HighResolutionTime start, RelativeTime period, RelativeTime cost, RelativeTime deadline, AsyncEventHandler overrunHandler, AsyncEventHandler missHandler) {
        
        super(cost, deadline, overrunHandler, missHandler);

        setPeriod(period);
        setStart(start);
        setDeadline(deadline);



    }

    /**
     * Gets the period.
     *
     * @return
     *      The current value in period.
     */
    public RelativeTime getPeriod() {

        return period;
    }

    /**
     * Gets the start time.
     *
     * @return
     *      The current value in start. This is the value that was specified in
     *      the constructor or by setStart(), not the actual absolute time
     *      corresponding to the start of one of the schedulable objects associated
     *      with this PeriodicParameters object.
     */
    public HighResolutionTime getStart() {

        return start;
    }

    /**
     * Sets the deadline value.
     *
     * If this parameter object is associated with any schedulable object
     * (by being passed through the schedulable object's constructor or set
     * with a method such as
     * RealtimeThread.setReleaseParameters(ReleaseParameters)) the deadline of
     * those schedulable objects is altered as specified by each schedulable
     * object's respective scheduler.
     *
     * Overrides:
     *          setDeadline in class ReleaseParameters
     *
     * @param deadline
     *          The latest permissible completion time measured from the
     * release time of the associated invocation of the schedulable object.
     * If deadline is null, the deadline is set to a new instance of
     * RelativeTime equal to period.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the time value of deadline is less than or equal to
     * zero, or if the new value of this deadline is incompatible with the
     * scheduler for any associated schedulable object.
     *
     * @throws IllegalAssignmentError
     *          Thrown if deadline cannot be stored in this.
     */
    public void setDeadline(RelativeTime deadline) {



        if (deadline == null) {
            deadline = period;
        }
        super.setDeadline(deadline);

    }

    /**
     * This method first performs a feasibility analysis using the new period,
     * cost and deadline attributes as replacements for the matching attributes
     * of this.
     *
     * If the resulting system is feasible the method replaces the current
     * attributes of this. If this parameter object is associated with any
     * schedulable object (by being passed through the schedulable object's
     * constructor or set with a method such as
     * RealtimeThread.setReleaseParameters(ReleaseParameters)) the parameters
     * of those schedulable objects are altered as specified by each
     * schedulable object's respective scheduler.
     *
     * @param period
     *          The proposed period. There is no default value. If period is
     * null an exception is thrown.
     *
     * @param cost
     *          The proposed cost. If null, the default value is a new instance
     * of RelativeTime(0,0).
     *
     * @param deadline
     *          The proposed deadline. If null, the default value is new
     * instance of RelativeTime(period).
     *
     * @returns
     *          True, if the resulting system is feasible and the changes are
     * made. False, if the resulting system is not feasible and no changes
     * are made.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the period is null or its time value is not greater
     * than zero, or if the time value of cost is less than zero, or if the
     * time value of deadline is not greater than zero. Also thrown if the
     * values are incompatible with the scheduler for any of the schedulable
     * objects which are presently using this parameter object.
     *
     * @throws IllegalAssignmentError
     *          Thrown if period, cost or deadline cannot be stored in this.
     */
    public boolean setIfFeasible(RelativeTime period, RelativeTime cost, RelativeTime deadline) {

        throw new UnsupportedOperationException("Not overrided yet.");
    }

    /**
     * Sets the period.
     *
     * @param period
     *          The value to which period is set.
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the given period is null or its time value is not
     * greater than zero. Also thrown if period is incompatible with the
     * scheduler for any associated schedulable object.
     *
     * @throws IllegalAssignmentError
     *          Thrown if period cannot be stored in this.
     */
    public void setPeriod(RelativeTime period) {
        if (period == null) {
            throw new IllegalArgumentException("Period must not be null.");
        }

        this.period = period;
    }

    /**
     * Sets the start time.
     *
     * The effect of changing the start time for any schedulable objects
     * associated with this parameter object is determined by the scheduler
     * associated with each schedulable object.
     *
     * Note: An instance of PeriodicParameters may be shared by several
     * schedulable objects. A change to the start time may take effect on a
     * subset of these schedulable objects. That leaves the start time returned
     * by getStart unreliable as a way to determine the start time of a
     * schedulable object.
     *
     * @param start
     *          The new start time. If null, the default value is a new
     * instance of RelativeTime(0,0).
     *
     * @throws java.lang.IllegalArgumentException
     *          Thrown if the given start time is incompatible with the
     * scheduler for any of the schedulable objects which are presently using
     * this parameter object.
     *
     * @throws IllegalAssignmentError
     *          Thrown if start cannot be stored in this.
     *
     */
    public void setStart(HighResolutionTime start) {
        if (start == null) {
            start = new RelativeTime(0, 0);
        }
        this.start = start;
    }
}
