package lejos.realtime;

/**
 *
 * @author Team iRboT
 */
public abstract class Clock {

    private static final Clock dfClock = new DefaultClock();

    public Clock(){
        
    }

    /**
     * Returns the relative time of the offset of the epoch of this clock from
     * the Epoch.
     *
     * For the real-time clock it will return a RelativeTime value equal to 0.
     * An UnsupportedOperationException is thrown if the clock does not support
     * the concept of date.
     *
     * @return
     *      A newly allocated RelativeTime object in the current execution
     * context with the offset past the Epoch for this clock. The returned
     * object is associated with this clock.
     *
     * @throws UnsupportedOperationException
     *          Thrown if the clock does not have the concept of date.
     */
    public abstract RelativeTime getEpochOffset();

    /**
     * There is always at least one clock object available: the system
     * real-time clock. This is the default Clock.
     *
     * @return
     *      The singleton instance of the default Clock
     */
    public static Clock getRealtimeClock(){
        return dfClock;
    }

    /**
     * Gets the resolution of the clock, the nominal interval between ticks.
     *
     * @return
     *      A newly allocated RelativeTime object in the current execution
     * context representing the resolution of this. The returned object is
     * associated with this clock.
     */
    public abstract RelativeTime getResolution();
        

    /**
     * Gets the current time in a newly allocated object.
     *
     * Note: This method will return an absolute time value that represents the
     * clock's notion of an absolute time. For clocks that do not measure
     * calendar time this absolute time may not represent a wall clock time.
     *
     * @return
     *      A newly allocated instance of AbsoluteTime in the current
     * allocation context, representing the current time. The returned object
     * is associated with this clock.
     */
    public abstract AbsoluteTime getTime();

    /**
     * Gets the current time in an existing object.
     *
     * The time represented by the given AbsoluteTime is changed at some time
     * between the invocation of the method and the return of the method.
     * Note: This method will return an absolute time value that represents the
     * clock's notion of an absolute time. For clocks that do not measure
     * calendar time this absolute time may not represent a wall clock time.
     *
     * @param dest
     *          The instance of AbsoluteTime object which will be updated in
     * place. The clock association of the dest parameter is ignored. When dest
     * is not null the returned object is associated with this clock. If dest
     * is null, then nothing happens.
     *
     * @return
     *      The instance of AbsoluteTime passed as parameter, representing the
     * current time, associated with this clock, or null if dest was null.
     */
    public abstract AbsoluteTime getTime(AbsoluteTime dest);

    /**
     * Set the resolution of this.
     *
     * For some hardware clocks setting resolution is impossible and if this
     * method is called on those clocks, then an UnsupportedOperationException
     * is thrown.
     *
     * @param resolution
     *          The new resolution of this, if the requested value is supported
     * by this clock. If resolution is smaller than the minimum resolution
     * supported by this clock then it throws IllegalArgumentException. If the
     * requested resolution is not available and it is larger than the minimum
     * resolution, then the clock will be set to the closest resolution that
     * the clock supports, via truncation. The value of the resolution
     * parameter is not altered. The clock association of the resolution
     * parameter is ignored.
     *
     * @throws IllegalArgumentException
     *          Thrown if resolution is null, or if the requested resolution is
     * smaller than the minimum resolution supported by this clock.
     *
     * @throws UnsupportedOperationException
     *          Thrown if the clock does not support setting its resolution.
     */
    public abstract void setResolution(RelativeTime resolution);
        
}
