package lejos.addon.gps;


/**
 * This class models data extracted from NMEA GSV Sentence
 * 
 * $GPGSV,1,1,13,02,02,213,,03,-3,000,,11,00,121,,14,13,172,05*67
 * 
 * 4    = SV PRN number
 * 5    = Elevation in degrees, 90 maximum
 * 6    = Azimuth, degrees from true north, 000 to 359
 * 7    = SNR, 00-99 dB (null when not tracking)
 * 
 * You can find out more about a satellite by looking up the PRN number here:
 * http://en.wikipedia.org/wiki/List_of_GPS_satellite_launches 
 * 
 * @author Juan Antonio Brenha Moral
 *
 */
public class Satellite{
	private int PRN = 0;
	private int elevation = 0;
	private int azimuth = 0;
	private int SNR = 0;
	private boolean tracked;
	
	/*
	 * Constructors
	 */

	public Satellite(){
		//
	}
	
	/**
	 * Constructor which indicate information about:
	 * PRN, Elevation, Azimuth and SNR
	 * 
	 * @param p
	 * @param e
	 * @param a
	 * @param s
	 */
	public Satellite(int p, int e, int a, int s){
		PRN = p;
		elevation = e;
		azimuth = a;
		SNR = s;
	}

	// TODO: getVehicleNumber() should be in here. SVN = Satellite Vehicle Number 
	
	/*
	 * Getters & Setters 
	 */

	/**
	 * Return PRN number from a Satellite. PRN stands for Pseudo-Random Noise. The PRN is the
	 * index to the assigned pseudorandom number sequence used to encode the satellite transmissions.
	 * 
	 * PRN is a binary signal with random noise-like properties. It is generated by mathematical algorithm or "code", 
	 * and consists of repeated pattern of 1's and 0's. This binary code can be modulated on the GPS carrier 
	 * waves using Binary Shift-Key (BSK) modulation. The C/A-Code and the P-Code are examples of PRN codes. 
	 * Each satellite transmits a unique C/A-Code and P-Code sequence (on the same L1 and L2 frequencies), 
	 * and hence a satellite may be identified according to its "PRN number", e.g. PRN2 or PRN14 are particular 
	 * GPS satellites.
	 * 
	 */
	public int getPRN(){
		return PRN;
	}
	
	/**
	 * Set PRN
	 * 
	 * @param p
	 */
	public void setPRN(int p){
		PRN = p;
	}

	/**
	 * How many degrees over the horizon the satellite is. This is a value between 0 and 90.
	 * 0 means it is by the horizon and 90 straight up.
	 * 
	 * @return Elevation in degrees (0 to 90)
	 */
	public int getElevation(){
		return elevation;
	}
	
	/**
	 * Indicates if the GPS receiver is actively tracking this satellite and using it to fix the GPS location.
	 * @return true means this satellite is part of the GPS solution. false means it is probably obscured/unable to receive the signal.
	 */
	public boolean isTracked() {
		return tracked;
	}
	
	public void setTracked(boolean tracked) {
		this.tracked = tracked;
	}
	
	/**
	 * Set Elevation
	 * 
	 * @param e
	 */
	public void setElevation(int e){
		elevation = e;
	}

	/**
	 * Direction to the satellite in degrees. This is a value between 0 and 360. 
	 * 0 is straight north, 90 is east, 180 is south and 270 is west. 
	 * Any value in between is of course also valid.
	 * 
	 * @return Azimuth in degrees, 0 to 360.
	 */
	  
	public int getAzimuth(){
		return azimuth;
	}
	
	/**
	 * Set Azimuth
	 * 
	 * @param a
	 */
	public void setAzimuth(int a){
		azimuth = a;
	}

	/**
	 * * Signal to Noise Ratio (SNR) for the signal from a satellite. This is a number between 0 and 99. 
	 * Where 99 is perfect and 0 means the satellite is not available. A typical good value is 40.
	 * And a GPS typically starts using a satellite when its SNR value is higher than 25.
	 * 
	 * @return SNR value between 0 and 99.
	 */
	 
	public int getSignalNoiseRatio(){
		return SNR;
	}
	
	/**
	 * Set SNR
	 * 
	 * @param s
	 */
	public void setSignalNoiseRatio(int s){
		SNR = s;
	}
}
