import java.util.*;
import java.io.*;
import java.nio.charset.*;
import java.net.*;

/** 
 * An interactive UDP chat program
 * @author chilowi at u-pem.fr
 */
public class UDPChat
{
	public static final int RECEIVE_TIMEOUT = 1000;
	public static final int BUFFER_LEN = 1024;
	
	private final MulticastSocket socket;
	private final byte[] buffer = new byte[BUFFER_LEN];
	private final DatagramPacket packet;
	private final Charset charset;
	private final Scanner sc;
	// InetSocketAddress is a useful class that combines an InetAddress with a port (it defines the coordinates of a socket)
	private List<InetSocketAddress> addressees = new ArrayList<InetSocketAddress>();
	// We store all the multicast groups we have subscribed
	private List<InetAddress> subscribedGroups = new ArrayList<InetAddress>();
	
	public UDPChat(int localPort, Charset charset) throws IOException
	{
		this.socket = new MulticastSocket(localPort);
		this.packet = new DatagramPacket(buffer, buffer.length);
		// We configure the receive timeout for the socket
		this.socket.setSoTimeout(RECEIVE_TIMEOUT);
		this.charset = charset;
		// Interact with the user using the standard input
		this.sc = new Scanner(System.in);
	}
	
	public void configure() throws IOException
	{
		System.out.println("Welcome on UDPChat!");
		System.out.println("Let's configure...");
		System.out.println("Enter the addressees (one by line following the pattern host/port): ");
		String line = null;
		do {
			line = sc.nextLine();
			if (! line.equals(""))
			{
				String[] components = line.split("/");
				String host = null;
				int port = -1;
				try {
					host = components[0];
					port = Integer.parseInt(components[1]);
				} catch (Exception e)
				{
					System.err.println(line + " is an invalid addressee");
					host = null;
				}
				if (host != null)
					addressees.add(new InetSocketAddress(host, port));
			}
		} while (! line.equals(""));
		System.out.println("Thanks. The configured addressees are " + addressees);
		System.out.println("Now, please indicate the multicast groups to join");
		do {
			line = sc.nextLine();
			if (! line.equals(""))
			{
				InetAddress ia = null;
				try {
					ia = InetAddress.getByName(line);
					socket.joinGroup(ia);
				} catch (Exception e)
				{
					ia = null;
					System.out.println("Sorry but the multicast address " + line + " does not seem valid");
				}
				if (ia != null) subscribedGroups.add(ia);
			}
		} while (! line.equals(""));
		System.out.println("Thank you. The configured multicast groups to listen are " + subscribedGroups);
		System.out.println("End of configuration. You can receive/send UDP messages");
	}
	
	public void communicate() throws IOException
	{
		// We alternate receive and send moments
		boolean go = true;
		while (go)
		{
			System.out.println("Waiting at most " + socket.getSoTimeout() + " ms for a new packet");
			boolean receive = true;
			while (receive)
			{
				try {
					String message = receive();
					System.out.println(message);
				} catch (SocketTimeoutException e)
				{
					receive = false; 
					// A timeout has just occurred, we exit from the receive phase
				}
			}
			boolean send = true;
			while (send)
			{
				System.out.println("Please enter a message to send (or a blank line to jump to the receive phase)");
				if (! sc.hasNextLine())
				{
					// The user has closed the standard input, we return from the method
					send = false;
					go = false;
				}
				if (send)
				{
					String line = sc.nextLine();
					if (line.equals("")) send = false; // Jum to the receive phase
					else
					{
						send(line);
					}
				}
			}
		}
		System.out.println("The end of communication.");
	}
	
	public String receive() throws IOException
	{
		packet.setData(buffer);
		socket.receive(packet);
		String message = new String(packet.getData(), 0, packet.getLength(), charset);
		return packet.getAddress() + ":" + packet.getPort() + "> " + message;
	}
	
	public void send(String message) throws IOException
	{
		packet.setData(message.getBytes(charset));
		for (InetSocketAddress isa: addressees)
		{
			// Send to each addressee the message
			packet.setAddress(isa.getAddress());
			packet.setPort(isa.getPort());
			socket.send(packet);
		}
	}
	
	public void close() throws IOException
	{
		System.err.println("Closing the socket");
		for (InetAddress group: subscribedGroups)
			socket.leaveGroup(group);
		socket.close();
	}
	
	public static void main(String[] args) throws IOException
	{
		int localPort = -1;
		Charset charset = null;
		try {
			localPort = Integer.parseInt(args[0]);
			charset = Charset.forName(args[1]); // We check that the charset exists (otherwise we have an exception)
		} catch (Exception e)
		{
			System.out.println("Error while interpreting the arguments.");
			System.out.println("Usage: java UDPChat bindPort charset");
		}
		UDPChat u = new UDPChat(localPort, charset);
		try {
			u.configure();
			u.communicate();
		} finally
		{
			// We do not forget to leave the groups and close the socket
			u.close();
		}
	}
}
				
		
		
