package fr.upem.jacosa.net;

import java.util.*;
import java.io.*;
import java.nio.charset.*;
import java.net.*;
import java.util.function.*;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.stream.*;

/** 
 * An interactive anagram game server.
 * 
 * @author chilowi at u-pem.fr
 * @license Apache 2
 */
public class AnagramServer
{
	public static final int BUFFER_LEN = 1024;
	public static final String CHARSET = "UTF-8";
	public static final int SESSION_TIMEOUT = 600; // in seconds
	
	private final DatagramSocket socket;
	private final byte[] buffer = new byte[BUFFER_LEN];
	private final DatagramPacket packet;
	
	private final Random rng = new Random();
	
	private final String[] dictionary;
	
	private final Map<SocketAddress, Session> sessionMap = new HashMap<>();
	
	/** Transform a string to a bag of chars */
	public static Map<Character, Integer> computeCharBag(String s)
	{
		Map<Character, Integer> m = new HashMap<>();
		for (int i = 0; i < s.length(); i++)
			m.merge(s.charAt(i), 1, (x, y) -> x + y);
		return m;
	}
	
	public static int compareCharBag(Map<Character, Integer> haystack, Map<Character, Integer> needle)
	{
		if (haystack.equals(needle)) return 2; // equal bags
		for (Map.Entry<Character, Integer> entry: needle.entrySet())
			if (haystack.getOrDefault(entry.getKey(), 0) < entry.getValue())
				return 0; // needle is not included in haystack
		return 1; // needle is included in haystack
	}
	
	public String createAnagram()
	{
		String word = dictionary[rng.nextInt(dictionary.length)];
		System.err.println("Create anagram for " + word);
		char[] wordChars = word.toCharArray();
		// shuffle the array
		for (int i = 0; i < wordChars.length; i++)
		{
			int j = rng.nextInt(wordChars.length - i);
			char tmp = wordChars[i];
			wordChars[i] = wordChars[i + j];
			wordChars[i + j] = tmp;
		}
		return new String(wordChars);
	}
	
	
	/** Game session */
	class Session
	{
		public final SocketAddress playerAddress;
		public final String anagram;
		private final Map<Character, Integer> charBag;
		private long lastTimestamp = System.nanoTime();
		
		public Session(SocketAddress playerAddress, String anagram)
		{
			this.playerAddress = playerAddress;
			this.anagram = anagram;
			this.charBag = computeCharBag(anagram);
			sessionMap.put(playerAddress, this);
			System.err.println("Create session for " + playerAddress + " with anagram " + anagram);
		}
		
		public Session(SocketAddress playerAddress)
		{
			// find an anagram in the dictionary for the game session
			this(playerAddress, createAnagram());
		}
		
		public void send(String message) throws IOException
		{
			packet.setData(message.getBytes(CHARSET));
			packet.setSocketAddress(playerAddress);
			socket.send(packet);
			lastTimestamp = System.nanoTime();
		}
		
		public void sendAnagram() throws IOException
		{
			send("anagram:" + anagram);
		}
		
		public void makeProposition(String proposition) throws IOException
		{
			boolean closing = false;
			proposition = proposition.trim().toLowerCase();
			int cmp = compareCharBag(charBag, computeCharBag(proposition));
			String prefix = "ko:";
			if (cmp == 1 && Arrays.binarySearch(dictionary, proposition) >= 0) // partial solution
				prefix = "ok:";
			else if (cmp == 2 && Arrays.binarySearch(dictionary, proposition) >= 0) // ultimate solution
			{
				prefix = "OK:";
				closing = true;
			}
			send(prefix + proposition);
			if (closing) close();
		}
		
		public void close()
		{
			sessionMap.remove(playerAddress);
			System.err.println("Closing session with " + playerAddress);
		}
	}
		
		
	
	public AnagramServer(int localPort, String[] dictionary) throws IOException
	{
		this.socket = new DatagramSocket(localPort);
		this.packet = new DatagramPacket(buffer, buffer.length);
		this.dictionary = dictionary;
	}
	
	public void loop(Supplier<Boolean> interrupter) throws IOException
	{
		while (! interrupter.get())
		{
			handleReceival();
			cleanSessions();
		}
	}
	
	public void handleReceival() throws IOException
	{
		packet.setData(buffer);
		socket.receive(packet);
		String message = new String(packet.getData(), 0, packet.getLength(), CHARSET);
		Session s = sessionMap.get(packet.getSocketAddress());
		if (s != null)
			s.makeProposition(message);
		else if (message.trim().equals("startGame"))
			new Session(packet.getSocketAddress()).sendAnagram();
		else
			System.err.println("Invalid request from " + packet.getSocketAddress() + ": " + message);
	}
	
	public void cleanSessions()
	{
		long now = System.nanoTime();
		sessionMap.values().stream().filter(x -> (now - x.lastTimestamp) / 1_000_000_000 > SESSION_TIMEOUT).collect(Collectors.toList()).forEach( x -> x.close() );
	}
	
	public void close() throws IOException
	{
		System.err.println("Closing the socket");
		socket.close();
	}
	
	public static void main(String[] args) throws IOException
	{
		int localPort = -1;
		String[] dictionary = null;
		try {
			localPort = Integer.parseInt(args[0]);
			dictionary = Files.lines(Paths.get(args[1])).map( x -> x.trim().toLowerCase() ).sorted().toArray( (n) -> new String[n] );
		} catch (Exception e)
		{
			System.out.println("Error while interpreting the arguments.");
			System.out.println("Usage: java AnagramServer port dictionaryPath");
		}
		AnagramServer u = new AnagramServer(localPort, dictionary);
		try {
			u.loop(() -> false); // never stop the server!
		} finally
		{
			u.close();
		}
	}
}
				
		
		
