import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.locks.*;

/** A multi-threaded message displayer
 * each word of the message is shown by a separate thread
 * all the subtlety is to display each message in the appropriate order ;)
 *
 * If the ORDERED constant is set to false, there are no waits and notifications (and the order is undefined).
 * 
 * @author chilowi at u-pem . fr
 */
public class ThreadDisplayer
{
	public static final boolean ORDERED = true;
	
	public static final Random random = new Random();
	
	private final String[] words;
	private final Lock lock;
	private final Condition[] conditions;
	private int currentIndex = 0;
	
	public ThreadDisplayer(String[] words)
	{
		this.words = words;
		this.lock = new ReentrantLock();
		this.conditions = new Condition[words.length];
		for (int i=1; i < conditions.length; i++)
			conditions[i] = lock.newCondition();
	}
	
	public void display()
	{
		// Create the threads
		for (int i =0; i < words.length; i++)
		{
			final int index = i;
			new Thread(new Runnable()
			{
				@Override
				public void run()
				{
					// Sleep a random amount of time
					// it will randomize the order of the displayed words if ORDER is set to false
					try {
						Thread.sleep(random.nextInt(500));
					} catch (InterruptedException e)
					{
						return;
					}
					// Now we acquire the lock and wait for the moment
					// to display our word
					lock.lock();
					try {
						if (ORDERED && index > 0)
							// Wait for the previous word to be displayed
							while (currentIndex < index)
								try {
									conditions[index].await();
								} catch (InterruptedException e)
								{
									System.out.print("interrupted");
									return;
								}
						if (index > 0)
							System.out.print(" "); // Print a separator
						System.out.print(words[index]);
						if (index == words.length - 1)
							System.out.println(""); // Print a line return after the last element
						// Update the state
						currentIndex = index+1;
						if (ORDERED && index < words.length - 1)
							// Notify the next thread
							conditions[index + 1].signal();
					} finally {
						lock.unlock();
					}
				}
			}).start();
		}
	}
	
	public static void main(String[] args)
	{
		new ThreadDisplayer(args).display();
	}
}
