package fr.upem.jacosa.collections;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Function;

// @run: LRUCacheTest

/** Interface for a cache map that keeps the Least Recently Used entries */
interface LRUCache<K, V>
{
	/** Get the value linked to key or insert it in the cache by computing it with the computer */
	public V get(K key, Function<K, V> computer);
	
	/** Number of values that has been computed */
	public int getMisses();
}

/** Default implementation of the LRU cache */
class LRUCacheImpl<K, V> implements LRUCache<K, V>
{
	private final int capacity;
	private LinkedHashMap<K,V> map = new LinkedHashMap<>();
	private int missCounter = 0;
	
	public LRUCacheImpl(int capacity)
	{
		this.capacity = capacity;
	}
	
	@Override
	public int getMisses()
	{
		return missCounter;
	}
	
	@Override
	public V get(K key, Function<K, V> computer)
	{
		V result = map.remove(key); // remove from the map
		if (result == null) 
		{
			result = computer.apply(key);
			missCounter++;
		}
		
		map.put(key, result); // put in the map (key will now be in last position in the LinkedHashMap)
		
		// check if we are in over-capacity
		// in this case we remove the first elements of the map
		int toRemove = map.size() - capacity;
		if (toRemove > 0)
			for (Iterator<Map.Entry<K, V>> it = map.entrySet().iterator(); it.hasNext() && toRemove > 0; )
			{
				it.next();
				it.remove();
				toRemove--;
			}
			
		return result;
	}
}

public class LRUCacheTest
{
	public static void main(String[] args)
	{
		LRUCacheImpl<Integer, Integer> cache = new LRUCacheImpl<>(5);
		for (int i = 0; i < 100; i++)
			System.out.println(cache.get(i, x -> x * x ));
		for (int i = 99; i >= 0; i--)
			System.out.println(cache.get(i, x -> x * x ));
		System.out.println("Number of misses: " + cache.getMisses());
	}
}