package fr.upem.jacosa.collections;

import java.util.*;
import java.lang.management.*;

/** A benchmark about the Collection.contains() method 
 * @author chilowi at u-pem.fr
 */
public class ContainsBench
{
	private final Random random;
	private final Collection<WrappedInt> collection;
	private final boolean badHash;
	
	/** An object containing an int */
	public static class WrappedInt implements Comparable<WrappedInt>
	{
		public final int value;
		
		public WrappedInt(int value) { this.value = value; }
		
		/** A method declared final can be overriden in child classes */
		@Override
		public final boolean equals(Object obj)
		{
			if (! (obj instanceof WrappedInt)) return false;
			return value == ((WrappedInt)obj).value;
		}
		
		@Override
		public final int compareTo(WrappedInt other)
		{
			if (this == other) return 0;
			if (other == null) return 1;
			if (this.value > other.value) return 1;
			if (this.value < other.value) return -1;
			return 0;
		}
		
		@Override
		public int hashCode() { return value; }
	}
	
	/** A WrappedInt with a dumb hashCode method */
	public static class BadWrappedInt extends WrappedInt
	{
		/** We declare again the constructor */
		public BadWrappedInt(int value) { super(value); }
	
		@Override 
		public int hashCode() { return 0; }
	}
	
	public ContainsBench(Collection<WrappedInt> collection, boolean badHash)
	{
		this.random = new Random();
		this.collection = collection;
		this.badHash = badHash;
	}
	
	public ContainsBench(Collection<WrappedInt> collection)
	{
		this(collection, false);
	}
	
	private WrappedInt createInt(int i)
	{
		return (badHash)?new BadWrappedInt(i):new WrappedInt(i);
	}
	
	/** We put n integers from 0 to n-1 in the collection */
	private void populateCollection(int n)
	{
		for (int i = 0; i < n; i++)
			collection.add(createInt(i));
	}
	
	/** We run the benchmark */
	public void run(int n, int k)
	{
		long startTime = getUserTime();
		populateCollection(n);
		System.out.println(String.format("Time required to populate %s with %d values %s: %f s", collection.getClass(), n, (badHash)?"with bad hash":"", (double)(getUserTime() - startTime) / 10e9));
		startTime = getUserTime();
		for (int i = 0; i < k; i++)
		{
			WrappedInt searched = createInt(random.nextInt(n));
			boolean c = collection.contains(searched);
			assert(c); // The searched value is always in the collection
		}
		System.out.println(String.format("Time required to run %d random contains() on %s of size %d %s: %f s", k, collection.getClass(), collection.size(), (badHash)?"with bad hash":"", (double)(getUserTime() - startTime) / 10e9));
	}
	
	public long getUserTime( ) 
	{
		ThreadMXBean bean = ManagementFactory.getThreadMXBean( );
		return bean.isCurrentThreadCpuTimeSupported( ) ? bean.getCurrentThreadUserTime( ) : 0L;
	}
	
	public static void main(String[] args)
	{
		int n = Integer.parseInt(args[0]);
		int k = Integer.parseInt(args[1]);
		
		// Test a LinkedList
		new ContainsBench(new LinkedList<WrappedInt>()).run(n, k);
		// Test an ArrayList
		new ContainsBench(new ArrayList<WrappedInt>()).run(n, k);
		// Test a HashSet with a good hash function
		new ContainsBench(new HashSet<WrappedInt>(), false).run(n, k);
		// Test a HashSet with a bad hash function
		new ContainsBench(new HashSet<WrappedInt>(), true).run(n, k);
		// Test a TreeSet
		new ContainsBench(new TreeSet<WrappedInt>()).run(n, k);
	}
}
	
