import { Injectable } from '@angular/core';
import { Observable, Observer } from 'rxjs';

@Injectable({
  providedIn: 'root'
})
export class TimestampService {

  getTimestamp(): number {
	return Math.floor((new Date().getTime()) / 1000);
  }

  timestampSubscriber() {
	const observers: Set<Observer<number>> = new Set(); // an array when we store all the subscribed observers
	let intervalHandle: any = null; // no interval scheduled yet
	let that = this; // reference to the service
	return (observer) => {
		if (intervalHandle === null) {
			intervalHandle = setInterval(() => {
				// action done at each period
				// we inform the observers about the timestamp
				// by calling their next method
				observers.forEach((observer) => observer.next(that.getTimestamp()));
			}, 1000);
		}
		observers.add(observer);
		// we must return an object with a method allowing the observer to unsubscribe
		return {
			unsubscribe() {
				observers.delete(observer); // remove the observer from the set
				if (observers.size == 0) { // if the set is empty, we can clear the interval
					clearInterval(intervalHandle);
					intervalHandle = null;
				}
			}
		}
	};
  }

  readonly timestampObservable = new Observable(this.timestampSubscriber());
  
  subscribe(observer) {
	this.timestampObservable.subscribe(observer);
  }

  constructor() { }
}
