import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';

import { Observable } from 'rxjs';
import { flatMap } from 'rxjs/operators';

const PICTURE_FETCHER_URL = "https://picsum.photos/$width/$height?random";

/** Make the service globally reachable as a singleton */
@Injectable({
  providedIn: 'root'
})
/** A service that fetches a random picture using a public web API */
export class PictureFetcherService {
	constructor(private httpClient: HttpClient) { }
	
	getPictureURL(width: number, height: number) {
		return PICTURE_FETCHER_URL.replace("$width", ""+width).replace("$height", ""+height);
	}
	
	/** This method converts a blob to an image (under the form of a data URL) with a FileReader */
	convertBlobToDataURL(blob: Blob): Observable<string> {
		return new Observable<string>( observer => {
			let reader = new FileReader();
			reader.addEventListener("load", () => { if (typeof(reader.result) === 'string') observer.next(reader.result) }, false);
			reader.readAsDataURL(blob);
		});
	}
	
	/** Return an Observable<Blob> with the JPEG picture */
	getPictureBlobObservable(width: number, height: number): Observable<Blob> {
		return this.httpClient.get(this.getPictureURL(width, height), { responseType: 'blob' });
	}
	
	/** Return an Observable<string> with the data URL of the picture */
	getPictureDataURLObservable(width: number, height: number): Observable<string> {
		return this.getPictureBlobObservable(width, height).pipe( flatMap( blob => this.convertBlobToDataURL(blob) ) );
	}
}
