import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';

import { Observable, of } from 'rxjs';
import { map, catchError } from 'rxjs/operators';

interface BitcoinQuote {
	ticker: { base: string, target: string, price: string, volume: string, change: string },
	timestamp: number,
	success: boolean,
	error: string
}

export class SimpleBitcoinQuote {
	constructor(public baseCurrency: string,
		public targetCurrency: string,
		public price: number,
		public date: Date) {}
}

@Injectable()
export class BitcoinTickerService {
  readonly API_URL = "https://api.cryptonator.com/api/ticker/btc-%target%";
  
  readonly CACHE_LIFE = 60; // in seconds, if we have a result younger, we do not query the API
  
  /** Store here the last known results */
  lastResults = new Map<string, SimpleBitcoinQuote>();
  
  constructor(private httpClient: HttpClient) { }
  
  /** Tell if a SimpleBitcoinQuote is expired or not */
  isFresh(quote: SimpleBitcoinQuote): boolean {
	return quote.date.getTime() + this.CACHE_LIFE * 1000 < new Date().getTime();
  }
  
  newQuoteObservable(targetCurrency: string): Observable<SimpleBitcoinQuote> {
    // maybe there is a recent result (aged from less than CACHE_LIFE seconds)
    let lastResult = this.lastResults.get(targetCurrency);
    if (lastResult instanceof SimpleBitcoinQuote && this.isFresh(lastResult))
		return of(lastResult); // no need to query again the service
    const url = this.API_URL.replace("%target%", targetCurrency.toLowerCase());
	return this.httpClient.get<BitcoinQuote>(url).pipe(
	    // we transform the BitcoinQuote to a SimpleBitcoinQuote
		map( data => {
			if (! data.success)
				throw new Error(`An error was returned by the API: data.error`);
			let result = new SimpleBitcoinQuote(
				data.ticker.base, 
				data.ticker.target, 
				parseFloat(data.ticker.price), 
				new Date(data.timestamp * 1000));
			this.lastResults.set(result.targetCurrency, result);
			return result;
		   }));
  }
}
